/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.snapshot;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.HLogLink;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.snapshot.SnapshotDescriptionUtils;
import org.apache.hadoop.hbase.snapshot.SnapshotReferenceUtil;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

@InterfaceAudience.Public
@InterfaceStability.Evolving
public final class SnapshotInfo
extends Configured
implements Tool {
    private static final Log LOG = LogFactory.getLog(SnapshotInfo.class);
    private FileSystem fs;
    private Path rootDir;
    private HTableDescriptor snapshotTableDesc;
    private HBaseProtos.SnapshotDescription snapshotDesc;
    private Path snapshotDir;

    public int run(String[] args) throws IOException, InterruptedException {
        String snapshotName = null;
        boolean showSchema = false;
        boolean showFiles = false;
        boolean showStats = false;
        for (int i = 0; i < args.length; ++i) {
            String cmd = args[i];
            try {
                if (cmd.equals("-snapshot")) {
                    snapshotName = args[++i];
                    continue;
                }
                if (cmd.equals("-files")) {
                    showFiles = true;
                    continue;
                }
                if (cmd.equals("-stats")) {
                    showStats = true;
                    continue;
                }
                if (cmd.equals("-schema")) {
                    showSchema = true;
                    continue;
                }
                if (cmd.equals("-h") || cmd.equals("--help")) {
                    this.printUsageAndExit();
                    continue;
                }
                System.err.println("UNEXPECTED: " + cmd);
                this.printUsageAndExit();
                continue;
            }
            catch (Exception e) {
                this.printUsageAndExit();
            }
        }
        if (snapshotName == null) {
            System.err.println("Missing snapshot name!");
            this.printUsageAndExit();
            return 1;
        }
        Configuration conf = this.getConf();
        this.fs = FileSystem.get((Configuration)conf);
        this.rootDir = FSUtils.getRootDir(conf);
        if (!this.loadSnapshotInfo(snapshotName)) {
            System.err.println("Snapshot '" + snapshotName + "' not found!");
            return 1;
        }
        this.printInfo();
        if (showSchema) {
            this.printSchema();
        }
        if (showFiles || showStats) {
            this.printFiles(showFiles);
        }
        return 0;
    }

    private boolean loadSnapshotInfo(String snapshotName) throws IOException {
        this.snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshotName, this.rootDir);
        if (!this.fs.exists(this.snapshotDir)) {
            LOG.warn((Object)("Snapshot '" + snapshotName + "' not found in: " + this.snapshotDir));
            return false;
        }
        this.snapshotDesc = SnapshotDescriptionUtils.readSnapshotInfo(this.fs, this.snapshotDir);
        this.snapshotTableDesc = FSTableDescriptors.getTableDescriptor(this.fs, this.snapshotDir);
        return true;
    }

    private void printInfo() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        System.out.println("Snapshot Info");
        System.out.println("----------------------------------------");
        System.out.println("   Name: " + this.snapshotDesc.getName());
        System.out.println("   Type: " + (Object)((Object)this.snapshotDesc.getType()));
        System.out.println("  Table: " + this.snapshotDesc.getTable());
        System.out.println(" Format: " + this.snapshotDesc.getVersion());
        System.out.println("Created: " + df.format(new Date(this.snapshotDesc.getCreationTime())));
        System.out.println();
    }

    private void printSchema() {
        System.out.println("Table Descriptor");
        System.out.println("----------------------------------------");
        System.out.println(this.snapshotTableDesc.toString());
        System.out.println();
    }

    private void printFiles(final boolean showFiles) throws IOException {
        if (showFiles) {
            System.out.println("Snapshot Files");
            System.out.println("----------------------------------------");
        }
        final String table = this.snapshotDesc.getTable();
        final SnapshotStats stats = new SnapshotStats(this.getConf(), this.fs, this.snapshotDesc);
        SnapshotReferenceUtil.visitReferencedFiles(this.fs, this.snapshotDir, new SnapshotReferenceUtil.FileVisitor(){

            @Override
            public void storeFile(String region, String family, String hfile) throws IOException {
                SnapshotStats.FileInfo info = stats.addStoreFile(region, family, hfile);
                if (showFiles) {
                    System.out.printf("%8s %s/%s/%s/%s %s%n", info.isMissing() ? "-" : StringUtils.humanReadableInt((long)info.getSize()), table, region, family, hfile, info.inArchive() ? "(archive)" : (info.isMissing() ? "(NOT FOUND)" : ""));
                }
            }

            @Override
            public void recoveredEdits(String region, String logfile) throws IOException {
                SnapshotStats.FileInfo info = stats.addRecoveredEdits(region, logfile);
                if (showFiles) {
                    System.out.printf("%8s recovered.edits %s on region %s%n", StringUtils.humanReadableInt((long)info.getSize()), logfile, region);
                }
            }

            @Override
            public void logFile(String server, String logfile) throws IOException {
                SnapshotStats.FileInfo info = stats.addLogFile(server, logfile);
                if (showFiles) {
                    System.out.printf("%8s log %s on server %s %s%n", info.isMissing() ? "-" : StringUtils.humanReadableInt((long)info.getSize()), logfile, server, info.isMissing() ? "(NOT FOUND)" : "");
                }
            }
        });
        System.out.println();
        if (stats.isSnapshotCorrupted()) {
            System.out.println("**************************************************************");
            System.out.printf("BAD SNAPSHOT: %d hfile(s) and %d log(s) missing.%n", stats.getMissingStoreFilesCount(), stats.getMissingLogsCount());
            System.out.println("**************************************************************");
        }
        System.out.printf("%d HFiles (%d in archive), total size %s (%.2f%% %s shared with the source table)%n", stats.getStoreFilesCount(), stats.getArchivedStoreFilesCount(), StringUtils.humanReadableInt((long)stats.getStoreFilesSize()), Float.valueOf(stats.getSharedStoreFilePercentage()), StringUtils.humanReadableInt((long)stats.getSharedStoreFilesSize()));
        System.out.printf("%d Logs, total size %s%n", stats.getLogsCount(), StringUtils.humanReadableInt((long)stats.getLogsSize()));
        System.out.println();
    }

    private void printUsageAndExit() {
        System.err.printf("Usage: bin/hbase %s [options]%n", ((Object)((Object)this)).getClass().getName());
        System.err.println(" where [options] are:");
        System.err.println("  -h|-help                Show this help and exit.");
        System.err.println("  -snapshot NAME          Snapshot to examine.");
        System.err.println("  -files                  Files and logs list.");
        System.err.println("  -stats                  Files and logs stats.");
        System.err.println("  -schema                 Describe the snapshotted table.");
        System.err.println();
        System.err.println("Examples:");
        System.err.println("  hbase " + ((Object)((Object)this)).getClass() + " \\");
        System.err.println("    -snapshot MySnapshot -files");
        System.exit(1);
    }

    public static SnapshotStats getSnapshotStats(Configuration conf, HBaseProtos.SnapshotDescription snapshot) throws IOException {
        Path rootDir = FSUtils.getRootDir(conf);
        FileSystem fs = FileSystem.get((Configuration)conf);
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshot, rootDir);
        final SnapshotStats stats = new SnapshotStats(conf, fs, snapshot);
        SnapshotReferenceUtil.visitReferencedFiles(fs, snapshotDir, new SnapshotReferenceUtil.FileVisitor(){

            @Override
            public void storeFile(String region, String family, String hfile) throws IOException {
                stats.addStoreFile(region, family, hfile);
            }

            @Override
            public void recoveredEdits(String region, String logfile) throws IOException {
                stats.addRecoveredEdits(region, logfile);
            }

            @Override
            public void logFile(String server, String logfile) throws IOException {
                stats.addLogFile(server, logfile);
            }
        });
        return stats;
    }

    static int innerMain(String[] args) throws Exception {
        return ToolRunner.run((Configuration)HBaseConfiguration.create(), (Tool)new SnapshotInfo(), (String[])args);
    }

    public static void main(String[] args) throws Exception {
        System.exit(SnapshotInfo.innerMain(args));
    }

    public static class SnapshotStats {
        private int hfileArchiveCount = 0;
        private int hfilesMissing = 0;
        private int hfilesCount = 0;
        private int logsMissing = 0;
        private int logsCount = 0;
        private long hfileArchiveSize = 0L;
        private long hfileSize = 0L;
        private long logSize = 0L;
        private final HBaseProtos.SnapshotDescription snapshot;
        private final Configuration conf;
        private final FileSystem fs;

        SnapshotStats(Configuration conf, FileSystem fs, HBaseProtos.SnapshotDescription snapshot) {
            this.snapshot = snapshot;
            this.conf = conf;
            this.fs = fs;
        }

        public HBaseProtos.SnapshotDescription getSnapshotDescription() {
            return this.snapshot;
        }

        public boolean isSnapshotCorrupted() {
            return this.hfilesMissing > 0 || this.logsMissing > 0;
        }

        public int getStoreFilesCount() {
            return this.hfilesCount + this.hfileArchiveCount;
        }

        public int getArchivedStoreFilesCount() {
            return this.hfileArchiveCount;
        }

        public int getLogsCount() {
            return this.logsCount;
        }

        public int getMissingStoreFilesCount() {
            return this.hfilesMissing;
        }

        public int getMissingLogsCount() {
            return this.logsMissing;
        }

        public long getStoreFilesSize() {
            return this.hfileSize + this.hfileArchiveSize;
        }

        public long getSharedStoreFilesSize() {
            return this.hfileSize;
        }

        public long getArchivedStoreFileSize() {
            return this.hfileArchiveSize;
        }

        public float getSharedStoreFilePercentage() {
            return (float)this.hfileSize / (float)(this.hfileSize + this.hfileArchiveSize) * 100.0f;
        }

        public long getLogsSize() {
            return this.logSize;
        }

        FileInfo addStoreFile(String region, String family, String hfile) throws IOException {
            String table = this.snapshot.getTable();
            Path path = new Path(family, HFileLink.createHFileLinkName(table, region, hfile));
            HFileLink link = new HFileLink(this.conf, path);
            boolean inArchive = false;
            long size = -1L;
            try {
                inArchive = this.fs.exists(link.getArchivePath());
                if (inArchive) {
                    size = this.fs.getFileStatus(link.getArchivePath()).getLen();
                    this.hfileArchiveSize += size;
                    ++this.hfileArchiveCount;
                } else {
                    size = link.getFileStatus(this.fs).getLen();
                    this.hfileSize += size;
                    ++this.hfilesCount;
                }
            }
            catch (FileNotFoundException e) {
                ++this.hfilesMissing;
            }
            return new FileInfo(inArchive, size);
        }

        FileInfo addRecoveredEdits(String region, String logfile) throws IOException {
            Path rootDir = FSUtils.getRootDir(this.conf);
            Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(this.snapshot, rootDir);
            Path path = SnapshotReferenceUtil.getRecoveredEdits(snapshotDir, region, logfile);
            long size = this.fs.getFileStatus(path).getLen();
            this.logSize += size;
            ++this.logsCount;
            return new FileInfo(true, size);
        }

        FileInfo addLogFile(String server, String logfile) throws IOException {
            HLogLink logLink = new HLogLink(this.conf, server, logfile);
            long size = -1L;
            try {
                size = logLink.getFileStatus(this.fs).getLen();
                this.logSize += size;
                ++this.logsCount;
            }
            catch (FileNotFoundException e) {
                ++this.logsMissing;
            }
            return new FileInfo(false, size);
        }

        static class FileInfo {
            private final boolean inArchive;
            private final long size;

            FileInfo(boolean inArchive, long size) {
                this.inArchive = inArchive;
                this.size = size;
            }

            public boolean inArchive() {
                return this.inArchive;
            }

            public boolean isMissing() {
                return this.size < 0L;
            }

            public long getSize() {
                return this.size;
            }
        }
    }
}

